﻿//----------------------------------------------------------
// Copyright (C) Microsoft Corporation. All rights reserved.
//----------------------------------------------------------
// TFS.UI.Controls.js
//
/*globals _comma_separated_list_of_variables_*/
/// <reference path="~/Scripts/DevTime.js" />
/// <reference path=~/Scripts/jquery-1.6.2-vsdoc.js" />
/// <reference path="~/Scripts/MicrosoftAjax.js"/>
/// <reference path="~/Scripts/TFS/TFS.js" />
/// <reference path="~/Scripts/TFS/TFS.Core.js" />
/// <reference path="~/Scripts/TFS/TFS.Diag.js" />
/// <reference path="~/Scripts/TFS/TFS.UI.js" />

TFS.module("TFS.UI.Controls", ["TFS.Resources.Presentation", "TFS.UI", "TFS.Diag", "TFS.Core", "TFS.Core.Utils"], function () {
    var log = TFS.Diag.log,
        getErrorMessage = TFS.getErrorMessage,
        Diag = TFS.Diag,
        Core = TFS.Core,
        assert = TFS.Diag.assert,
        verbose = TFS.Diag.LogVerbosity.Verbose,
        delegate = TFS.Core.delegate,
        Resources = TFS.Resources.Presentation,
        Utils = TFS.Core.Utils,
        getId,
        domElem = TFS.UI.domElem;

    getId = function () {
        var idSeed = 0;
        return function () {
            return (++idSeed);
        };
    }();


    function Enhancement(options) {
        if (this.getType() === Enhancement) {
            throw new Error("You cannot instantiate an abstract type.");
        }

        this.getTypeName();
        this._options = $.extend({ earlyInitialize: true, enabled: true }, options);
    }

    Enhancement.extend(function () {
        var enhancementList = [];
        return {
            optionsPrefix: "",
            getType: function () {
                return this;
            },
            getTypeName: function () {
                return this._typeName || ("tfs.controls." + this.getName());
            },
            getEnhancementOptions: function (element) {
                var options, optionsElement, json;
                options = element.data(this.optionsPrefix + "options");

                if (!options) {
                    optionsElement = element.children("." + this.optionsPrefix + "options");

                    if (optionsElement.length > 0) {
                        json = optionsElement.html();

                        if (json) {
                            options = Utils.parseMSJSON(json, false);
                        }
                    }
                }

                return options;
            },
            enhance: function (element, options) {
                var control, type = this, $element;

                if (element instanceof Enhancement) {
                    $element = element.getElement();
                }
                else {
                    $element = $(element);
                }

                control = $element.data("enhancement:" + this.getTypeName());

                if (!control) {
                    if ($.isFunction(options)) {
                        options = options.call(type, $element);
                    }

                    options = $.extend(this.getEnhancementOptions($element), options);

                    control = new type(options);
                    control.enhance($element, element);

                    $element.data("enhancement:" + this.getTypeName(), control);
                }

                return control;
            },
            getInstance: function (element) {
                var type = this, instance;
                while (type) {
                    instance = element.data(type.getTypeName());

                    if (instance instanceof type) {
                        return instance;
                    }

                    type = type.base;
                }
            },
            getInstances: function (element) {
                var result = this.getInstance(element), type = this;
                if (!result) {
                    return element.find(':data(' + this.getTypeName() + ')').map(function () {
                        return type.getInstance($(this));
                    });
                }
                else {
                    return [result];
                }
            },
            registerEnhancement: function (selector, options, errorCallback) {
                var type = this, typeName = this.getTypeName();

                function enhance(context) {
                    var instances = [], selection;

                    if (context) {
                        selection = $(context);
                    }

                    if (!selection || !selection.is(selector)) {
                        selection = $(selector, selection);
                    }

                    selection.each(function () {
                        instances[instances.length] = type.enhance($(this), options);
                    });

                    return instances;
                }

                $(function () {
                    try {
                        enhance(document);
                        Diag.logTracePoint("Enhancement.registered-complete", [selector, typeName]);
                    } catch (e) {
                        e = Error.create(String.format("Enhanchement failed for '{0}'. Details: {1}", typeName, getErrorMessage(e)), { name: "EnhancementFailed", innerException: e });
                        if (errorCallback) {
                            TFS.handleError(e, errorCallback);
                        }
                        else {
                            throw e;
                        }
                    }
                });

                Diag.logTracePoint("Enhancement.registered-pending", [selector, typeName]);
                enhancementList[enhancementList.length] = { typeName: typeName, enhancement: enhance };
            },
            ensureEnhancements: function (context, errorCallback) {
                var enhancementEntry, i, l, instances = [];

                for (i = 0, l = enhancementList.length; i < l; i++) {
                    enhancementEntry = enhancementList[i];

                    if (this === Enhancement || enhancementEntry.typeName === this.getTypeName()) {
                        try {
                            instances = instances.concat(enhancementEntry.enhancement(context) || []);
                        }
                        catch (e) {
                            e = Error.create(String.format("Enhanchement failed for '{0}'. Details: {1}", enhancementEntry.typeName, getErrorMessage(e)), { name: "EnhancementFailed", innerException: e });
                            if (errorCallback) {
                                TFS.handleError(e, errorCallback);
                            }
                            else {
                                throw e;
                            }
                        }
                    }
                }

                return instances;
            },
            registerJQueryWidget: function (widgetName, widgetOptions) {
                var type = this;

                if (!widgetName) {
                    widgetName = this._widgetName;

                    if (!widgetName) {
                        assert(this.getTypeName() ? true : false, "_widgetName or _typeName needs to present in order to register this control as JQuery widget.");
                        widgetName = this.getTypeName().split(".");
                        widgetName = widgetName[widgetName.length - 1];
                    }
                }

                $.fn[widgetName] = function (options) {
                    return this.each(function () {
                        type.enhance($(this), $.extend(widgetOptions || {}, options));
                    });
                };
            }
        };
    }());

    Enhancement.prototype = {
        _id: null,
        _typeName: null,
        _eventNamespace: null,
        _options: null,
        _initialized: false,
        _element: null,
        _trackedElements: null,
        _delayedFunctions: null,
        _disposed: false,

        _getUniqueId: function () {
            return getId();
        },

        _getId: function () {
            if (!this._id) {
                this._setId(this._getUniqueId());
            }

            return this._id;
        },

        _setId: function (id) {
            this._id = id;
        },

        getTypeName: function () {
            if (!this._typeName) {
                this._typeName = this.getType().getTypeName();
            }

            return this._typeName;
        },

        _getEventNameSpace: function () {
            if (!this._eventNamespace) {
                this._eventNamespace = this.getTypeName().replace(/\./g, "_") + this._getId();
            }

            return this._eventNamespace;
        },

        getType: function () {
            return this.constructor;
        },

        initialize: function () {
            this._initialized = true;
        },

        _ensureInitialized: function () {
            if (!this._initialized) {
                this.initialize();

                return true;
            }

            return false;
        },

        _attemptInitialize: function () {
            if (this._options.earlyInitialize) {
                this.initialize();
            }
        },

        enhance: function ($element, originalElement) {
            this._originalElement = originalElement;
            this._enhance($element);
            this._attemptInitialize();
        },

        _enhance: function (element, wrap) {
            this._setElement(element);
        },

        _setElement: function (element) {
            var type = this.getType();

            if (this._element !== element) {
                this._cleanup();

                this._element = element;
                this._bind("remove.remove_" + this.getTypeName(), delegate(this, this._dispose));

                // Associating control with the element's data
                while (type) {
                    this._element.data(type.getTypeName(), this);
                    type = type.base;
                }

                this._setStyles();
            }
        },

        _setStyles: function () {
            var element = this._element;

            if (this._options.coreCssClass) {
                element.addClass(this._options.coreCssClass);
            }

            if (this._options.cssClass) {
                element.addClass(this._options.cssClass);
            }
        },

        getElement: function () {
            /// <summary>Gets the element associated with this control.</summary>
            /// <returns type="object" />

            return this._element;
        },

        _fire: function (element, eventType, args) {
            if (typeof element === "string") {
                args = eventType;
                eventType = element;
                element = this._element;
            }

            return element.trigger(eventType, args);
        },

        _bind: function (element, eventType, handler, track) {
            var $element, self = this, namespace;

            if (typeof element === "string") {
                track = handler;
                handler = eventType;
                eventType = element;
                element = this._element;
            }

            $element = $(element);
            if ($element.length > 0) {
                namespace = this._getEventNameSpace();
                $element.bind($.map(eventType.split(" "), function (et) {
                    return et + "." + namespace;
                }).join(" "),
                handler);

                if (track) {
                    $element.each(function () {
                        if (this !== self._element[0]) {
                            self._trackElement(this);
                        }
                    });
                }
            }

            return this;
        },

        _unbind: function (element, eventType, handler, track) {
            var $element, self = this, namespace;

            if (typeof element === "string") {
                handler = track;
                track = eventType;
                eventType = element;
                element = this._element;
            }

            $element = $(element);
            if ($element.length > 0) {
                namespace = this._getEventNameSpace();

                $element.unbind($.map(eventType.split(" "), function (et) {
                    return et + "." + namespace;
                }).join(" "), handler);

                if (track) {
                    $element.each(function () {
                        if (this !== self._element[0]) {
                            self._untrackElement(this);
                        }
                    });
                }
            }

            return this;
        },

        _trackElement: function (domElement) {
            var elems = this._trackedElements, i, l, record, found;
            if (!elems) {
                this._trackedElements = elems = [];
            }

            for (i = 0, l = elems.length; i < l; i++) {
                record = elems[i];
                if (record.elem === domElement) {
                    record.count++;
                    found = true;
                    break;
                }
            }

            if (!found) {
                elems.push({ elem: domElement, count: 1 });
            }
        },

        _untrackElement: function (domElement) {
            var elems = this._trackedElements, i, l, record, found = -1;
            if (elems) {
                for (i = 0, l = elems.length; i < l; i++) {
                    record = elems[i];
                    if (record.elem === domElement) {
                        record.count--;
                        found = i;
                        break;
                    }
                }

                if (found >= 0) {
                    if (!record.count) {
                        elems.splice(found);
                    }
                }
            }
        },

        delayExecute: function (name, msDelay, cancelPendingOps, func) {
            /// <summary>Executes the provided function after the specified amount of time</summary>
            /// <param name="name" type="String">(Optional) Name for this operation. Allows subsequent calls to cancel this action.</param>
            /// <param name="msDelay" type="Integer">Delay in milliseconds to wait before executing the Function</param>
            /// <param name="cancelPendingOps" type="Boolean">If true, cancel any pending requests with this name. If false, and there are outstanding requests with this name already in progress, then do nothing.</param>
            /// <param name="func" type="Function">Method to execute after the delay</param>

            var operation;

            if (!name) {
                // Unnamed operation - just use the Core delay method
                Core.delay(this, msDelay, func);
                return;
            }

            if (!this._delayedFunctions) {
                this._delayedFunctions = {};
            }

            operation = this._delayedFunctions[name];
            if (operation) {
                operation.setDelay(msDelay);
                operation.setMethod(this, func);
            }
            else {
                operation = new Core.DelayedFunction(this, msDelay, name, func);
                this._delayedFunctions[name] = operation;
            }

            if (cancelPendingOps) {
                operation.reset();
            }
            else {
                operation.start();
            }
        },

        cancelDelayedFunction: function (name) {
            /// <summary>Cancels any pending delayed functions (delayExecute calls) with the specified name</summary>
            /// <param name="name" type="String">Name (supplied in the delayExecute call) of the operations to cancel</param>
            /// <returns type="Boolean">True if any operation was canceled. False if no operations with the specified name were in progress</returns>
            var prevOperation;

            if (this._delayedFunctions) {
                prevOperation = this._delayedFunctions[name];
                if (prevOperation) {
                    prevOperation.cancel();
                    return true;
                }
            }

            return false;
        },

        _cleanup: function () {
            var type = this.getType();

            if (this._element) {
                this._unbind("remove.remove_" + this.getTypeName());
                this._element.unbind("." + this._getEventNameSpace());

                while (type) {
                    this._element.data(type.getTypeName(), null);
                    type = type.base;
                }
            }
        },

        _dispose: function () {
            var elems = this._trackedElements, i, l;

            if (elems) {
                for (i = 0, l = elems.length; i < l; i++) {
                    $(elems[i]).unbind("." + this._getEventNameSpace());
                }
            }

            this._trackedElements = null;
            this._disposed = true;
        },

        dispose: function () {
            this._cleanup();
            this._dispose();
        }
    };

    function BaseControl(options) {
        if (this.getType() === BaseControl) {
            throw new Error("You cannot instantiate an abstract type.");
        }

        this.baseConstructor.call(this, $.extend({ tagName: "div" }, options));
    }

    BaseControl.extend(function () {
        return {
            createIn: function (container, options) {
                var control = new this(options), $container;

                if (container instanceof Enhancement) {
                    $container = container.getElement();
                }
                else {
                    $container = $(container);
                }

                control.createIn($container);

                return control;
            }
        };
    }());

    BaseControl.inherit(Enhancement, {
        _getUniqueId: function () {
            var id;
            if (this._element) {
                id = this._element.attr("id");
            }

            if (typeof id === "undefined") {
                id = this._base();
            }

            return id;
        },
        _setId: function (id) {
            if (this._element) {
                this._element.attr("id", id);
            }

            this._base(id);
        },
        dispose: function () {
            this._base();
            if (this._element) {
                this._element.remove();
                this._element = null;
            }
        },
        showElement: function () {
            assert(this._element ? true : false, "DomElement is null or undefined.");
            this._element.show();
        },

        hideElement: function () {
            assert(this._element ? true : false, "DomElement is null or undefined.");
            this._element.hide();
        },

        enableElement: function (enabled) {
            if (enabled) {
                this._element.removeAttr("disabled");
                this._element.removeClass("disabled");
            }
            else {
                this._element.attr("disabled", "disabled");
                this._element.addClass("disabled");
            }
        },
        showBusyOverlay: function () {
            if (!this._overlay) {
                this._overlay = $("<div />").addClass("control-busy-overlay").appendTo(this._element.parent());
            }
            this._overlay.show();
        },
        hideBusyOverlay: function () {
            if (this._overlay) {
                this._overlay.hide();
            }
        },
        _createElement: function () {
            var element = $(domElem(this._options.tagName || "div"));

            this._setElement(element);
        },

        _initializeElement: function () {
            if (this._element) {
                if (this._options.id) {
                    this._setId(this._options.id);
                }
            }
        },

        _setStyles: function () {
            var element = this._element;

            if (this._options.width) {
                element.width(this._options.width);
            }

            if (this._options.height) {
                element.height(this._options.height);
            }

            if (this._options.title) {
                element.attr("title", this._options.title);
            }

            this._base();
        },

        createIn: function (container) {
            this._createIn(container);
            this._initializeElement();
            this._attemptInitialize();
        },

        _createIn: function (container) {
            this._createElement();
            container.append(this._element);
        },

        focus: function () {
            /// <summary>Set Focus to the control</summary>
            // TODO: override in child controls that have different focus receiver
            this._element.focus();
        },

        _fireChange: function (sender) {
            /// <summary>Fires the change event for the control either immediately or delayed</summary>
            /// <param name="sender" type="Object">Source element of the event</param>
            /// <param name="immediate" type="Boolean">(Optional) Specifies whether event should be fired immediately or not</param>
            sender = sender || this;

            // Actual event fire happening here
            if ($.isFunction(this._options.change)) {
                // Set sender as initial parameter if undefined
                arguments[0] = sender;
                arguments.length = arguments.length || 1;
                if (this._options.change.apply(sender, arguments) === false) {
                    return false;
                }
            }

            return this._fire("change", sender);
        }
    });

    function TreeNode(text, config, children) {
        this._ensureNodeId();
        this.text = text;
        this.config = config || {};
        this.children = [];
        this.addRange(children);
    }

    TreeNode.create = function (text, config, children) {
        return new TreeNode(text, config, children);
    };

    TreeNode.prototype = function () {
        var nodeIdSeed = 0;

        function treeNodeComparer(comparer) {
            comparer = comparer || String.localeIgnoreCaseComparer;
            return function (n1, n2) {
                return comparer(n1.text, n2.text);
            };
        }

        return {
            id: -1,
            root: false,
            text: null,
            parent: null,
            children: null,
            config: null,
            expanded: false,
            selected: false,
            icon: null,
            _ensureNodeId: function () {
                if (this.id < 0) {
                    this.id = nodeIdSeed++;
                }
            },
            hasChildren: function () {
                return this.children.length > 0;
            },
            clear: function () {
                this.children = [];
            },
            remove: function () {
                var that = this;
                if (this.parent) {
                    this.parent.children = $.map(this.parent.children, function (node) {
                        if (node !== that) {
                            return node;
                        }
                    })
                }
            },
            add: function (node) {
                this.children.push(node);
                node.parent = this;
            },
            addRange: function (nodes) {
                var i, l;
                if (nodes) {
                    for (i = 0, l = nodes.length; i < l; i++) {
                        this.add(nodes[i]);
                    }
                }
            },
            findNode: function (path, sepChar, comparer) {
                return Utils.findTreeNode.call(this, path, sepChar || "/", comparer || String.localeIgnoreCaseComparer, "text");
            },
            _sort: function (recursive, treeNodeComparer) {
                var i, l, children = this.children;
                if (children && children.length) {
                    children.sort(treeNodeComparer);

                    if (recursive) {
                        for (i = 0, l = children.length; i < l; i++) {
                            children[i]._sort(recursive, treeNodeComparer);
                        }
                    }
                }
            },
            sort: function (recursive, comparer) {
                this._sort(recursive, treeNodeComparer(comparer));
            },
            path: function (includeRoot, sepChar) {
                return Utils.calculateTreePath.call(this, includeRoot, sepChar || "/", "text", "root");
            },
            level: function (noRoot) {
                var level = 0, n = this.parent;

                while (n) {
                    if (noRoot && n.root) {
                        break;
                    }

                    level++;
                    n = n.parent;
                }

                return level;
            }
        };
    }();

    function BaseDataSource(options) {
        this._options = $.extend({
            sorted: false,
            comparer: String.localeIgnoreCaseComparer
        }, options);

        this.setSource(this._options.source);
    }

    BaseDataSource.prototype = {
        _options: null,
        _source: null,
        _items: null,
        _allItems: null,
        setSource: function (source) {
            this._source = source;
            this._items = null;
            this._allItems = null;
        },
        prepareSource: function (source) {
            var items = source || [];

            if (this._options.sorted) {
                Array.sortIfNotSorted(items, this._options.comparer || String.localeIgnoreCaseComparer);
            }

            this.setItems(items);
        },
        getComparer: function () {
            return this._options.comparer;
        },
        ensureItems: function () {
            var source;
            if (!this._items) {
                source = this._source;
                if ($.isFunction(source)) {
                    source = source.call(this);
                }

                this.prepareSource(source);
            }
        },
        getItems: function (all) {
            this.ensureItems();

            return all ? this._allItems : this._items;
        },
        setItems: function (items, allItems) {
            this._items = items || [];
            this._allItems = allItems || this._items;
        },
        getCount: function (all) {
            return this.getItems(all).length;
        },
        getItem: function (index, all) {
            return this.getItems(all)[index];
        },
        getItemText: function (index, all) {
            return this.getItem(index, all) + "";
        },
        getItemIndex: function (itemText, startsWith, all) {
            var i, l, hi, lo,
                itemTextLength,
                comparer = this._options.comparer,
                items,
                comparison,
                foundText;

            if (!itemText) {
                return -1;
            }

            items = this.getItems(all);

            if (this._options.sorted) {
                //source is sorted so do a binary search

                itemTextLength = itemText.length;
                lo = 0; hi = items.length - 1;

                while (hi >= lo) {
                    i = (lo + hi) >> 1; //mid point

                    foundText = this.getItemText(i, all);
                    comparison = comparer(itemText, foundText.substr(0, itemTextLength));

                    if (comparison < 0) {
                        hi = i - 1;
                    } else if (comparison > 0) {
                        lo = i + 1;
                    } else {
                        break;
                    }
                }

                if (comparison === 0) {
                    //there was a match

                    if (foundText.length === itemTextLength) {
                        return i; //found exact match
                    } else if (startsWith) {
                        //search up to get first match, this should happen during partial search
                        while (i > 0 && comparer(itemText, this.getItemText(i - 1, all).substr(0, itemTextLength)) === 0) {
                            i--;
                        }

                        return i;
                    }
                }
            }
            else {
                if (startsWith) {
                    for (i = 0, l = items.length; i < l; i++) {
                        if (String.startsWith(this.getItemText(i, all), itemText, comparer)) {
                            return i;
                        }
                    }
                }
                else {
                    for (i = 0, l = items.length; i < l; i++) {
                        if (comparer(this.getItemText(i, all), itemText) === 0) {
                            return i;
                        }
                    }
                }
            }

            return -1;
        },
        nextIndex: function (selectedIndex, delta, all) {
            var itemCount = this.getCount(all);

            if (itemCount < 1) {
                return -1;
            }
            else if (selectedIndex < 0) {
                return 0;
            }
            else if (delta > 0) {
                    //Next
                if (selectedIndex === (itemCount - 1)) {
                    return -1;
                }

                return Math.min(itemCount - 1, selectedIndex + delta);
            }
            else {
                //Prev
                if (selectedIndex === 0) {
                    return -1;
                }

                return Math.max(0, selectedIndex + delta);
            }
        }
    };

    function ListDataSource(options) {
        this.baseConstructor.call(this, $.extend({
        }, options));
    }

    ListDataSource.inherit(BaseDataSource, function () {
        return {
        };
    }());

    function TreeDataSource(options) {
        this.root = new TreeNode("");
        this.root.expanded = true;
        this.root.root = true;

        this.baseConstructor.call(this, $.extend({
            sepChar: "\\"
        }, options));
    }

    TreeDataSource.inherit(BaseDataSource, function () {
        function flatten(node, items, all) {
            var i, l, n, children = node.children;

            if (children && (l = children.length)) {
                for (i = 0; i < l; i++) {
                    n = children[i];
                    items[items.length] = n;

                    if (all || n.expanded) {
                        flatten(n, items, all);
                    }
                }
            }
        }

        function convertToTreeNodes(items) {
            return $.map(items, function (item) {
                var node = new TreeNode(item.text, item.config);

                if (item.children && item.children.length) {
                    node.addRange(convertToTreeNodes(item.children));
                }

                return node;
            });
        }

        function ensureSourceIsTreeNodes(source) {
            if (source) {
                if (source.length) {
                    if (!(source[0] instanceof TreeNode)) {
                        return convertToTreeNodes(source);
                    }
                }

                return source;
            }
            else {
                return [];
            }
        }

        function expandToLevel(node, level) {
            node.expanded = true;
            if (level && node.hasChildren()) {
                $.each(node.children, function (i, n) {
                    expandToLevel(n, level - 1);
                });
            }
        }

        return {
            root: null,
            setSource: function (source) {
                this.root.clear();
                this._base(source);
            },
            prepareSource: function (source) {
                var nodes = ensureSourceIsTreeNodes(source), items, allItems;

                this.root.addRange(nodes);

                if (this._options.sorted) {
                    this.root.sort(true, this._options.comparer);
                }

                if (this._options.treeLevel) {
                    expandToLevel(this.root, this._options.treeLevel);
                }

                allItems = [];
                flatten(this.root, allItems, true);

                items = [];
                flatten(this.root, items, false);

				this.setItems(items, allItems);
			},
			_prepareCurrentItems: function () {
				var items = [];
				flatten(this.root, items, false);
				this.setItems(items, this.getItems(true));
			},
			getItemText: function (index, all, textOnly) {
			    if (textOnly === true) {
			        return this.getItem(index, all).text;
			    }
			    else {
			        return this.getItem(index, all).path(false, this._options.sepChar);
			    }
			},
			getItemIndex: function (itemText, startsWith, all) {
				var index = this._base(itemText, startsWith, true), node, parent, reconstruct;

                if (index >= 0) {
                    node = this.getItem(index, true);

                    //node is found. Lets figure out if it is visible
                    while (node.parent) {
                        node = node.parent;
                        if (!node.expanded) {
                            reconstruct = true;
                        }

                        node.expanded = true; //make visible
                    }

                    if (reconstruct) {
                        //node was not visible, recreate items array
                        this._prepareCurrentItems();
                    }
                    // we need to call now with the same all setting so we get the right index in the tree taking collapsed sub trees into account
                    return this._base(itemText, startsWith, all);
                }

                return index;
            },
            expandNode: function (node) {
                if (!node.expanded) {
                    node.expanded = true;
                    this._prepareCurrentItems();
                }
            },
            collapseNode: function (node) {
                if (node.expanded) {
                    node.expanded = false;
                    this._prepareCurrentItems();
                }
            }
        };
    }());

    function VirtualizingListView(options) {
        this.baseConstructor.call(this, $.extend({
            coreCssClass: "virtualizing-list-view",
            maxRowCount: 8,
            selectedIndex: -1
        }, options));
    }

    VirtualizingListView.extend({
        _typeName: "tfs.virtualizingListView"
    });

    VirtualizingListView.inherit(BaseControl, function () {
        return {
            _itemsContainer: null,
            _scrollContainer: null,
            _scrollSpacer: null,
            _dataSource: null,
            visibleRowCount: 1,
            _firstVisible: 0,
            _selectedIndex: -1,
            _ignoreScrollEvent: false,
            _rowHeight: 20,
            _enableMouseOver: true,
            _prevMousePos: null,
            initialize: function () {
                this._bind("click", delegate(this, this._onClick));
                this._itemsContainer = $(domElem("ul", "items"));
                this._bind(this._itemsContainer, "mouseover", delegate(this, this._onMouseOver));
                this._bind(this._itemsContainer, "mousemove", delegate(this, this._onMouseMove));
                this._bind(this._itemsContainer, $.browser.mozilla ? "DOMMouseScroll" : "mousewheel", delegate(this, this._onMouseWheel));

                this._element.append(this._itemsContainer);

                this._scrollContainer = $(domElem("div", "scroller"));
                this._bind(this._scrollContainer, "scroll", delegate(this, this._onScroll));

                this._scrollSpacer = $(domElem("div", "content-spacer"));
                this._scrollContainer.append(this._scrollSpacer);

                this._element.append(this._scrollContainer);
                this._dataSource = this._options.dataSource;
                this._selectedIndex = this._options.selectedIndex;

                this.update();
                this._enableMouseOver = true;

                this._base();
            },
            update: function () {
                var count, height;
                count = this._dataSource.getCount();
                this.visibleRowCount = Math.min(count, this._options.maxRowCount);

                if (this._selectedIndex >= 0) {
                    this._selectedIndex = Math.min(this._selectedIndex, count - 1);
                }

                this._setVisibleBounds(this._selectedIndex);
                this._drawItems();
                height = this._itemsContainer.outerHeight();
                this._element.height(height);
                this._setupScrollbar(height);
            },
            _setVisibleBounds: function (visibleItemIndex) {
                visibleItemIndex = visibleItemIndex || 0;

                if (visibleItemIndex <= this._firstVisible) {
                    this._firstVisible = Math.max(0, visibleItemIndex);
                }
                else if (visibleItemIndex >= (this._firstVisible + this.visibleRowCount)) {
                    this._firstVisible = Math.max(0, Math.min(visibleItemIndex, this._dataSource.getCount() - this.visibleRowCount));
                }

                this._firstVisible = Math.max(0, Math.min(this._firstVisible, this._dataSource.getCount() - this.visibleRowCount));
            },
            scrollItemIntoView: function (index) {
                this._setVisibleBounds(index);
                this._drawItems();
                this._updateScrollbar();
            },
            _createItem: function (index) {
                var text = this._dataSource.getItemText(index) || "";
                return $(domElem("li", this._options.itemCss))
                        .text(text)
                        .attr("title", text);
            },
            _drawItems: function () {
                var i, start, end, createItem = this._options.createItem || this._createItem, item;

                this._itemsContainer.empty();
                start = this._firstVisible;
                end = Math.min(start + this.visibleRowCount, this._dataSource.getCount());

                for (i = start; i < end; i++) {
                    item = createItem.call(this, i);
                    item.data("index", i);

                    this._itemsContainer.append(item);
                }

                this._updateItemStyles();
                this._enableMouseOver = false;
            },
            _updateItemStyles: function () {
                var firstVisible = this._firstVisible, selectedIndex = this._selectedIndex;

                this._itemsContainer.children("li").each(function (i, item) {
                    $(this).toggleClass("selected", (i + firstVisible) === selectedIndex);
                })
            },
            _setupScrollbar: function (height) {
                var itemCount = this._dataSource.getCount(), needsScrollBars = this.visibleRowCount < itemCount;

                this._element.toggleClass("scroll", needsScrollBars);

                if (needsScrollBars) {
                    this._rowHeight = (height || this._itemsContainer.outerHeight()) / this.visibleRowCount;
                    this._scrollSpacer.height(this._rowHeight * itemCount);
                    this._updateScrollbar();
                }
            },
            _updateScrollbar: function () {
                try {
                    this._ignoreScrollEvent = true;
                    this._scrollContainer.scrollTop(this._rowHeight * this._firstVisible);
                } finally {
                    this._ignoreScrollEvent = false;
                }
            },
            _onScroll: function (e) {
                var itemCount = this._dataSource.getCount(), ratio;
                if (!this._ignoreScrollEvent) {
                    this._firstVisible = Math.max(0, Math.min(Math.floor(this._scrollContainer.scrollTop() / this._rowHeight), itemCount - this.visibleRowCount));
                    this._drawItems();
                }

                return false;
            },
            _onMouseMove: function (e) {
                var mousePos = e.screenX + "-" + e.screenY;

                //needed for IE. When itemcontainer is emptied and filled with new elements
                //mouse over event is get fired in IE
                if (!this._prevMousePos || this._prevMousePos !== mousePos) {
                    this._prevMousePos = mousePos;
                    this._enableMouseOver = true;
                }
            },
            _onMouseOver: function (e) {
                var $target = $(e.target), $li;

                if (this._enableMouseOver) {
                    this._enableMouseOver = false;

                    $li = $target.closest("li");
                    if ($li.length) {
                        this._selectedIndex = $li.data("index");
                        this._updateItemStyles();
                    }
                }
            },
            _onMouseWheel: function (e) {
                var delta, firstVisible;

                delta = TFS.UI.getWheelDelta(e);

                if (delta != 0) {
                    this._firstVisible = Math.max(0, Math.min(this._firstVisible + (delta > 0 ? -1 : 1), this._dataSource.getCount() - this.visibleRowCount));
                    this.scrollItemIntoView(this._firstVisible);
                    return false;
                }
            },
            _onClick: function (e) {
                var $target = $(e.target), $li, itemIndex;
                $li = $target.closest("li");

                if ($li.length) {
                    itemIndex = $li.data("index");

                    if (this._options.itemClick) {
                        if (this._options.itemClick.call(this, e, itemIndex, $target, $li) === false) {
                            return false;
                        }
                    }

                    this.setSelectedIndex(itemIndex, true);
                    this._updateItemStyles();
                    this._fireSelectionChanged(true);
                }
            },
            selectNext: function (page) {
                var selectedIndex = this._dataSource.nextIndex(this._selectedIndex, page ? this.visibleRowCount : 1);

                if (selectedIndex >= 0) {
                    this.setSelectedIndex(selectedIndex);
                    this._fireSelectionChanged();
                    return true;
                }

                return false;
            },
            selectPrev: function (page) {
                var selectedIndex = this._dataSource.nextIndex(this._selectedIndex, -(page ? this.visibleRowCount : 1));

                if (selectedIndex >= 0) {
                    this.setSelectedIndex(selectedIndex);
                    this._fireSelectionChanged();
                    return true;
                }

                return false;
            },
            _fireSelectionChanged: function (accept) {
                if (this._options.selectionChange) {
                    this._options.selectionChange.call(this, this._selectedIndex, accept);
                }
            },
            getSelectedIndex: function () {
                return this._selectedIndex;
            },
            setSelectedIndex: function (selectedIndex, noScrollIntoView) {
                if (this._selectedIndex !== selectedIndex) {
                    this._selectedIndex = selectedIndex;

                    if (!noScrollIntoView) {
                        this.scrollItemIntoView(selectedIndex);
                    }
                }
            }
        };
    }());

    return {
        getId: getId,
        Enhancement: Enhancement,
        BaseControl: BaseControl,
        TreeNode: TreeNode,
        BaseDataSource: BaseDataSource,
        ListDataSource: ListDataSource,
        TreeDataSource: TreeDataSource,
        VirtualizingListView: VirtualizingListView
    };
});


// SIG // Begin signature block
// SIG // MIIapwYJKoZIhvcNAQcCoIIamDCCGpQCAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFDXP5WM4SkP7
// SIG // OJrfWcftOU9+fIiJoIIVeTCCBLowggOioAMCAQICCmEC
// SIG // jkIAAAAAAB8wDQYJKoZIhvcNAQEFBQAwdzELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRp
// SIG // bWUtU3RhbXAgUENBMB4XDTEyMDEwOTIyMjU1OFoXDTEz
// SIG // MDQwOTIyMjU1OFowgbMxCzAJBgNVBAYTAlVTMRMwEQYD
// SIG // VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
// SIG // MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
// SIG // DTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIg
// SIG // RFNFIEVTTjpGNTI4LTM3NzctOEE3NjElMCMGA1UEAxMc
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCASIw
// SIG // DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAJbsjkdN
// SIG // VMJclYDXTgs9v5dDw0vjYGcRLwFNDNjRRi8QQN4LpFBS
// SIG // EogLQ3otP+5IbmbHkeYDym7sealqI5vNYp7NaqQ/56ND
// SIG // /2JHobS6RPrfQMGFVH7ooKcsQyObUh8yNfT+mlafjWN3
// SIG // ezCeCjOFchvKSsjMJc3bXREux7CM8Y9DSEcFtXogC+Xz
// SIG // 78G69LPYzTiP+yGqPQpthRfQyueGA8Azg7UlxMxanMTD
// SIG // 2mIlTVMlFGGP+xvg7PdHxoBF5jVTIzZ3yrDdmCs5wHU1
// SIG // D92BTCE9djDFsrBlcylIJ9jC0rCER7t4utV0A97XSxn3
// SIG // U9542ob3YYgmM7RHxqBUiBUrLHUCAwEAAaOCAQkwggEF
// SIG // MB0GA1UdDgQWBBQv6EbIaNNuT7Ig0N6JTvFH7kjB8jAf
// SIG // BgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7syuwwzWzDzBU
// SIG // BgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vY3JsLm1pY3Jv
// SIG // c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsGAQUFBwEBBEww
// SIG // SjBIBggrBgEFBQcwAoY8aHR0cDovL3d3dy5taWNyb3Nv
// SIG // ZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRUaW1lU3Rh
// SIG // bXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0G
// SIG // CSqGSIb3DQEBBQUAA4IBAQBz/30unc2NiCt8feNeFXHp
// SIG // aGLwCLZDVsRcSi1o2PlIEZHzEZyF7BLUVKB1qTihWX91
// SIG // 7sb1NNhUpOLQzHyXq5N1MJcHHQRTLDZ/f/FAHgybgOIS
// SIG // CiA6McAHdWfg+jSc7Ij7VxzlWGIgkEUvXUWpyI6zfHJt
// SIG // ECfFS9hvoqgSs201I2f6LNslLbldsR4F50MoPpwFdnfx
// SIG // Jd4FRxlt3kmFodpKSwhGITWodTZMt7MIqt+3K9m+Kmr9
// SIG // 3zUXzD8Mx90Gz06UJGMgCy4krl9DRBJ6XN0326RFs5E6
// SIG // Eld940fGZtPPnEZW9EwHseAMqtX21Tyi4LXU+Bx+BFUQ
// SIG // axj0kc1Rp5VlMIIE7DCCA9SgAwIBAgITMwAAALARrwqL
// SIG // 0Duf3QABAAAAsDANBgkqhkiG9w0BAQUFADB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQTAeFw0xMzAxMjQyMjMzMzla
// SIG // Fw0xNDA0MjQyMjMzMzlaMIGDMQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEB
// SIG // AQUAA4IBDwAwggEKAoIBAQDor1yiIA34KHy8BXt/re7r
// SIG // dqwoUz8620B9s44z5lc/pVEVNFSlz7SLqT+oN+EtUO01
// SIG // Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Unm/P4LZ5BNisL
// SIG // QPu+O7q5XHWTFlJLyjPFN7Dz636o9UEVXAhlHSE38Cy6
// SIG // IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpOoPXJCiHiquMB
// SIG // Nkf9L4JqgQP1qTXclFed+0vUDoLbOI8S/uPWenSIZOFi
// SIG // xCUuKq6dGB8OHrbCryS0DlC83hyTXEmmebW22875cHso
// SIG // AYS4KinPv6kFBeHgD3FN/a1cI4Mp68fFSsjoJ4TTfsZD
// SIG // C5UABbFPZXHFAgMBAAGjggFgMIIBXDATBgNVHSUEDDAK
// SIG // BggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGmWjNN2pgHgP+E
// SIG // Hr6H+XIyQfIwUQYDVR0RBEowSKRGMEQxDTALBgNVBAsT
// SIG // BE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRmYWYwYjcxLWFk
// SIG // MzctNGFhMy1hNjcxLTc2YmMwNTIzNDRhZDAfBgNVHSME
// SIG // GDAWgBTLEejK0rQWWAHJNy4zFha5TJoKHzBWBgNVHR8E
// SIG // TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NybC9wcm9kdWN0cy9NaWNDb2RTaWdQQ0Ff
// SIG // MDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUHAQEETjBMMEoG
// SIG // CCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NlcnRzL01pY0NvZFNpZ1BDQV8wOC0zMS0y
// SIG // MDEwLmNydDANBgkqhkiG9w0BAQUFAAOCAQEAMdduKhJX
// SIG // M4HVncbr+TrURE0Inu5e32pbt3nPApy8dmiekKGcC8N/
// SIG // oozxTbqVOfsN4OGb9F0kDxuNiBU6fNutzrPJbLo5LEV9
// SIG // JBFUJjANDf9H6gMH5eRmXSx7nR2pEPocsHTyT2lrnqkk
// SIG // hNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2yRPnwPJNtQtjo
// SIG // fOYXoJtoaAko+QKX7xEDumdSrcHps3Om0mPNSuI+5PNO
// SIG // /f+h4LsCEztdIN5VP6OukEAxOHUoXgSpRm3m9Xp5QL0f
// SIG // zehF1a7iXT71dcfmZmNgzNWahIeNJDD37zTQYx2xQmdK
// SIG // Dku/Og7vtpU6pzjkJZIIpohmgjCCBbwwggOkoAMCAQIC
// SIG // CmEzJhoAAAAAADEwDQYJKoZIhvcNAQEFBQAwXzETMBEG
// SIG // CgmSJomT8ixkARkWA2NvbTEZMBcGCgmSJomT8ixkARkW
// SIG // CW1pY3Jvc29mdDEtMCsGA1UEAxMkTWljcm9zb2Z0IFJv
// SIG // b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MB4XDTEwMDgz
// SIG // MTIyMTkzMloXDTIwMDgzMTIyMjkzMloweTELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWljcm9zb2Z0IENv
// SIG // ZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCycllcGTBkvx2aYCAgQpl2U2w+
// SIG // G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/3sJCTiPVcgDb
// SIG // NVcKicquIEn08GisTUuNpb15S3GbRwfa/SXfnXWIz6pz
// SIG // RH/XgdvzvfI2pMlcRdyvrT3gKGiXGqelcnNW8ReU5P01
// SIG // lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJpL9oZC/6SdCni
// SIG // di9U3RQwWfjSjWL9y8lfRjFQuScT5EAwz3IpECgixzdO
// SIG // PaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn0i1i8UU956wI
// SIG // APZGoZ7RW4wmU+h6qkryRs83PDietHdcpReejcsRj1Y8
// SIG // wawJXwPTAgMBAAGjggFeMIIBWjAPBgNVHRMBAf8EBTAD
// SIG // AQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGCNxUBBAUCAwEA
// SIG // ATAjBgkrBgEEAYI3FQIEFgQU/dExTtMmipXhmGA7qDFv
// SIG // pjy82C0wGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
// SIG // HwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8KuEKU5VZ5KQw
// SIG // UAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC5taWNy
// SIG // b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvbWljcm9z
// SIG // b2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUFBwEBBEgwRjBE
// SIG // BggrBgEFBQcwAoY4aHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNyb3NvZnRSb290Q2VydC5j
// SIG // cnQwDQYJKoZIhvcNAQEFBQADggIBAFk5Pn8mRq/rb0Cx
// SIG // MrVq6w4vbqhJ9+tfde1MOy3XQ60L/svpLTGjI8x8UJiA
// SIG // IV2sPS9MuqKoVpzjcLu4tPh5tUly9z7qQX/K4QwXacul
// SIG // nCAt+gtQxFbNLeNK0rxw56gNogOlVuC4iktX8pVCnPHz
// SIG // 7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y4k74jKHK6BOl
// SIG // kU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wdzaKMvSeyeWNW
// SIG // RKJRzfnpo1hW3ZsCRUQvX/TartSCMm78pJUT5Otp56mi
// SIG // LL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q70eFW6NB4lhhc
// SIG // yTUWX92THUmOLb6tNEQc7hAVGgBd3TVbIc6YxwnuhQ6M
// SIG // T20OE049fClInHLR82zKwexwo1eSV32UjaAbSANa98+j
// SIG // Zwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKNMxZlHg6K3RDe
// SIG // ZPRvzkbU0xfpecQEtNP7LN8fip6sCvsTJ0Ct5PnhqX9G
// SIG // uwdgR2VgQE6wQuxO7bN2edgKNAltHIAxH+IOVN3lofvl
// SIG // RxCtZJj/UBYufL8FIXrilUEnacOTj5XJjdibIa4NXJzw
// SIG // oq6GaIMMai27dmsAHZat8hZ79haDJLmIz2qoRzEvmtzj
// SIG // cT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq/2mbluIQqBC0
// SIG // N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZoNAAAAAAAHDAN
// SIG // BgkqhkiG9w0BAQUFADBfMRMwEQYKCZImiZPyLGQBGRYD
// SIG // Y29tMRkwFwYKCZImiZPyLGQBGRYJbWljcm9zb2Z0MS0w
// SIG // KwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
// SIG // ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1MzA5WhcNMjEw
// SIG // NDAzMTMwMzA5WjB3MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSEw
// SIG // HwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Ew
// SIG // ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCf
// SIG // oWyx39tIkip8ay4Z4b3i48WZUSNQrc7dGE4kD+7Rp9FM
// SIG // rXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr6Hu97IkHD/cO
// SIG // BJjwicwfyzMkh53y9GccLPx754gd6udOo6HBI1PKjfpF
// SIG // zwnQXq/QsEIEovmmbJNn1yjcRlOwhtDlKEYuJ6yGT1VS
// SIG // DOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd++NIT8wi3U21S
// SIG // tEWQn0gASkdmEScpZqiX5NMGgUqi+YSnEUcUCYKfhO1V
// SIG // eP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68eeEExd8yb3zuD
// SIG // k6FhArUdDbH895uyAc4iS1T/+QXDwiALAgMBAAGjggGr
// SIG // MIIBpzAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBQj
// SIG // NPjZUkZwCu1A+3b7syuwwzWzDzALBgNVHQ8EBAMCAYYw
// SIG // EAYJKwYBBAGCNxUBBAMCAQAwgZgGA1UdIwSBkDCBjYAU
// SIG // DqyCYEBWJ5flJRP8KuEKU5VZ5KShY6RhMF8xEzARBgoJ
// SIG // kiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJk/IsZAEZFglt
// SIG // aWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jvc29mdCBSb290
// SIG // IENlcnRpZmljYXRlIEF1dGhvcml0eYIQea0WoUqgpa1M
// SIG // c1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9taWNyb3NvZnRyb290Y2VydC5jcmwwVAYIKwYBBQUH
// SIG // AQEESDBGMEQGCCsGAQUFBzAChjhodHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY3Jvc29mdFJv
// SIG // b3RDZXJ0LmNydDATBgNVHSUEDDAKBggrBgEFBQcDCDAN
// SIG // BgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wDRDbd6bStd9vO
// SIG // eVFNAbEudHFbbQwTq86+e4+4LtQSooxtYrhXAstOIBNQ
// SIG // md16QOJXu69YmhzhHQGGrLt48ovQ7DsB7uK+jwoFyI1I
// SIG // 4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mRKiQicPv2/OR4
// SIG // mS4N9wficLwYTp2OawpylbihOZxnLcVRDupiXD8WmIsg
// SIG // P+IHGjL5zDFKdjE9K3ILyOpwPf+FChPfwgphjvDXuBfr
// SIG // Tot/xTUrXqO/67x9C0J71FNyIe4wyrt4ZVxbARcKFA7S
// SIG // 2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGysOUzU9nm/qhh6
// SIG // YinvopspNAZ3GmLJPR5tH4LwC8csu89Ds+X57H2146So
// SIG // dDW4TsVxIxImdgs8UoxxWkZDFLyzs7BNZ8ifQv+AeSGA
// SIG // nhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU2DKATCYqSCRf
// SIG // WupW76bemZ3KOm+9gSd0BhHudiG/m4LBJ1S2sWo9iaF2
// SIG // YbRuoROmv6pH8BJv/YoybLL+31HIjCPJZr2dHYcSZAI9
// SIG // La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCleKuzoJZ1GtmS
// SIG // hxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/Jmu5J4PcBZW+J
// SIG // C33Iacjmbuqnl84xKf8OxVtc2E0bodj6L54/LlUWa8kT
// SIG // o/0xggSaMIIElgIBATCBkDB5MQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQgQ29kZSBTaWdu
// SIG // aW5nIFBDQQITMwAAALARrwqL0Duf3QABAAAAsDAJBgUr
// SIG // DgMCGgUAoIG8MBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3
// SIG // AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
// SIG // MCMGCSqGSIb3DQEJBDEWBBTzEy/ZaLT9K5KilpDN41w8
// SIG // mr1uVzBcBgorBgEEAYI3AgEMMU4wTKAygDAAVABGAFMA
// SIG // LgBVAEkALgBDAG8AbgB0AHIAbwBsAHMALgBkAGUAYgB1
// SIG // AGcALgBqAHOhFoAUaHR0cDovL21pY3Jvc29mdC5jb20w
// SIG // DQYJKoZIhvcNAQEBBQAEggEAqxaCw7P1BDfD2UmI9HiJ
// SIG // ke9FC8NkAqCZivcbwhS8teJpy1NFi2CLtG1Oq2guLzqB
// SIG // M5XCGNm46Wniz/WwvxMzJocJN8+U2dil85+gfRBY1G42
// SIG // UjNBfZ496pGU9RWPjZJz4keD4Tyfaa0iGwIf5WYVkXm4
// SIG // VIu0n5O1Idj4fj2f7PH/UQjCtaOhcIig5Frqeo1CeFB/
// SIG // qJ3y3OOtzKqehFBnJh3gHTSEB43Y1bR0feLsUVN/jMdP
// SIG // plTBQHPdVEQOUQdN7QSlXOxqaFeivtSVUQh2bJ0nmqCn
// SIG // v5d2PatXh4TP2XcFGhxZS1FJk1RI+DZVk3gNZQh43xLq
// SIG // ayczEKQgC4C0NKGCAh8wggIbBgkqhkiG9w0BCQYxggIM
// SIG // MIICCAIBATCBhTB3MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSEw
// SIG // HwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EC
// SIG // CmECjkIAAAAAAB8wCQYFKw4DAhoFAKBdMBgGCSqGSIb3
// SIG // DQEJAzELBgkqhkiG9w0BBwEwHAYJKoZIhvcNAQkFMQ8X
// SIG // DTEzMDMxNTA2MzM1NVowIwYJKoZIhvcNAQkEMRYEFALw
// SIG // iWDVOJOotr9YjRcChEli5jCOMA0GCSqGSIb3DQEBBQUA
// SIG // BIIBACMEgEnXAw4TG8835/kHuUwutma5H5YxcMD7Nnqg
// SIG // EJxL+RLurchxUaDBX6rVqJI57FOdKeob1ZH5LuqwlV7i
// SIG // sVSN9A4s0yzevZGwiGTKadVDlOpTlyjZLOK3cldwYQur
// SIG // IfDA7T70h1jmIq8tDGuZFvDyi2lAH/stJo3XKcjCElg7
// SIG // DEWA0EBue0EQj4gyYg2vAi3m0aj2XXHrijVPG9mDzcOi
// SIG // ouBdx+61SEWeKh4Z/6VZfy+lPYQASmEbZw2Mfw70uLZO
// SIG // 840dB+Qp7Zjt8Xu1dP6WE4wT00DJH5Aa3De8ObzyXfOB
// SIG // pelw/SMqumZvnnVlzpjhB9kjiQVVvemrF5ZTevs=
// SIG // End signature block
